const fs = require('fs');
const path = require('path');

// Helper to parse simple markdown (bold, newlines)
function parseMarkdown(text) {
  if (!text) return '';
  // 1. Handle bold: **text** -> <strong>text</strong>
  let html = text.replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>');
  // 2. Handle newlines: \n -> <br>
  html = html.replace(/\n/g, '<br>');
  return html;
}

// Helper to read premium content if it exists
function getPremiumContent(serviceId) {
  const contentPath = path.join(__dirname, '../content/services', `${serviceId}-premium.json`);
  if (fs.existsSync(contentPath)) {
    return JSON.parse(fs.readFileSync(contentPath, 'utf8'));
  }
  return null;
}

// Helper to generate placeholder premium content from basic service data
function generatePlaceholderContent(service) {
  return {
    serviceId: service.id,
    serviceName: service.name,
    metaTitle: `${service.name} Services Near You - 24/7 Emergency Help`,
    metaDescription: `Connect with local ${service.name.toLowerCase()} professionals. Available 24/7 for emergencies in your area.`,
    h1: `24/7 Emergency ${service.name} Services`,
    heroIntro: `When you have a ${service.name.toLowerCase()} emergency, every minute counts. Rapid Service Connect helps you find independent, local professionals who can respond quickly to your urgent needs.`,
    sections: {
      commonProblems: {
        title: `Common ${service.name} Problems`,
        intro: `Our network of professionals handles a wide range of ${service.name.toLowerCase()} issues, including:`,
        problems: service.emergencies.map(e => ({
          title: e,
          description: `Professional assistance for ${e.toLowerCase()} to prevent further damage and restore safety.`
        }))
      },
      whyEmergency: {
        title: `Why ${service.name} Issues Can Be Emergencies`,
        content: `Delaying repairs for ${service.name.toLowerCase()} problems can lead to significant property damage, safety hazards, and increased repair costs. Immediate professional attention is often required to stabilize the situation.`
      },
      howItWorks: {
        title: "How Rapid Service Connect Works",
        content: "1. **Choose Your Service**: Select the specific help you need.\n2. **Call to Connect**: Use our dedicated number to reach a professional.\n3. **Get Help**: Speak directly with a local provider to arrange service."
      },
      regionalVariation: {
        title: "Service Availability",
        content: `We connect you with ${service.name.toLowerCase()} professionals across major US cities, understanding that local climate and infrastructure can affect your specific needs.`
      },
      whenToCallImmediately: {
        title: "When to Call Immediately",
        intro: "Contact a professional right away if you experience:",
        signs: [
          "Immediate safety risks",
          "Rapidly spreading damage",
          "System failure during extreme weather",
          "Health hazards"
        ],
        outro: "Don't wait when safety is at risk."
      },
      whatToExpect: {
        title: "What to Expect",
        content: "When you call, you'll be connected with an independent professional who can assess your situation, provide an estimated arrival time, and discuss potential costs."
      },
      costFactors: {
        title: "Cost Considerations",
        content: "Emergency service costs can vary based on the time of day, severity of the issue, and required parts. Your service provider can offer more specific pricing information."
      },
      preventionTips: {
        title: "Prevention Tips",
        intro: "Reduce the risk of future emergencies with these tips:",
        tips: service.tips || [
          "Regular maintenance checks",
          "Addressing small issues promptly",
          "Knowing your system's age and condition"
        ],
        outro: "Proactive care saves money in the long run."
      }
    },
    faqs: [
      {
        question: `Is ${service.name.toLowerCase()} available 24/7?`,
        answer: "Yes, we connect you with professionals who offer 24/7 emergency response."
      },
      {
        question: `How fast can a ${service.name.toLowerCase()} pro arrive?`,
        answer: "Response times vary by location and availability, but emergency providers prioritize urgent calls."
      },
      {
        question: `Do you provide the service directly?`,
        answer: "No, Rapid Service Connect is a referral service connecting you with independent local professionals."
      }
    ]
  };
}

function generatePremiumServicePages(services, cities, OUTPUT_DIR, ensureDir) {
  const { getHeader, getFooter, getNavigation, getDisclaimer } = require('./seo-utils');

  services.forEach(service => {
    // Try to get premium content, otherwise generate placeholder
    const content = getPremiumContent(service.slug) || generatePlaceholderContent(service);

    // Generate Breadcrumbs
    const breadcrumbs = [
      { name: "Home", url: "/" },
      { name: "Services", url: "/services/" },
      { name: service.name, url: `/services/${service.slug}/` }
    ];

    // Generate FAQ Schema
    const faqSchema = {
      "@context": "https://schema.org",
      "@type": "FAQPage",
      "mainEntity": content.faqs.map(faq => ({
        "@type": "Question",
        "name": faq.question,
        "acceptedAnswer": {
      < div class= "container" >
        <div class="hero-badge">24/7 Emergency Service</div>
        <h1>${content.h1}</h1>
        <p class="hero-subtitle">${content.heroIntro}</p>
        <div class="hero-cta-group">
          <a href="/contact/" class="btn btn-primary btn-large">📞 Call Now for Help</a>
          <a href="#cities" class="btn btn-secondary btn-large">View Cities We Serve</a>
        </div>
        <div class="trust-badges">
          <span>⚡ Available 24/7</span>
          <span>✓ Verified Local Pros</span>
          <span>🔒 Independent Referral</span>
        </div>
      </div >
    </section >

    < !--Breadcrumbs -->
    <div class="container" style="margin-top: var(--space-4);">
      <nav class="breadcrumbs" aria-label="Breadcrumb">
        <ol>
          ${breadcrumbs.map((crumb, idx) => `<li${idx === breadcrumbs.length - 1 ? ' aria-current="page"' : ''}><a href="${crumb.url}">${crumb.name}</a></li>`).join('')}
        </ol>
      </nav>
    </div>

    <!--Common Problems Section-- >
    <section class="section-premium">
      <div class="container">
        <div class="section-header">
          <h2>${content.sections.commonProblems.title}</h2>
          <div class="section-divider"></div>
          <p>${content.sections.commonProblems.intro}</p>
        </div>
        <div class="steps-premium"> <!-- Reusing grid layout -->
          ${content.sections.commonProblems.problems.map(p => `
            <div class="content-card">
              <h3>${p.emoji ? `<span style="font-size: 1.5em;">${p.emoji}</span>` : ''} ${p.title}</h3>
              <p>${p.description}</p>
            </div>
          `).join('')}
        </div>
      </div>
    </section>

    <!--Emergency Context & Urgency-- >
    <section class="section-premium" style="background: var(--color-gray-50);">
      <div class="container">
        <div class="content-wrapper">
          <h2>${content.sections.whyEmergency.title}</h2>
          <p>${parseMarkdown(content.sections.whyEmergency.content)}</p>
          
          <div class="emergency-banner">
            <h3>${content.sections.whenToCallImmediately.title}</h3>
            <p>${content.sections.whenToCallImmediately.intro}</p>
            <ul style="list-style: none; padding: 0; margin: var(--space-4) 0;">
              ${content.sections.whenToCallImmediately.signs.map(sign => `<li>⚠️ ${sign}</li>`).join('')}
            </ul>
            <p><strong>${content.sections.whenToCallImmediately.outro}</strong></p>
          </div>
        </div>
      </div>
    </section>

    <!--How It Works-- >
    <section class="section-premium">
      <div class="container">
        <div class="section-header">
          <h2>${content.sections.howItWorks.title}</h2>
          <div class="section-divider"></div>
        </div>
        <div class="steps-premium">
          <div class="step-card-premium">
            <div class="step-number-premium">1</div>
            <h3>Choose Your Service</h3>
            <p>Select the specific emergency service you need from our comprehensive list.</p>
          </div>
          <div class="step-card-premium">
            <div class="step-number-premium">2</div>
            <h3>Call to Connect</h3>
            <p>Dial our dedicated emergency number to be routed to a local professional.</p>
          </div>
          <div class="step-card-premium">
            <div class="step-number-premium">3</div>
            <h3>Get Help Fast</h3>
            <p>Speak directly with the provider to arrange immediate service dispatch.</p>
          </div>
        </div>
        <div class="cta-block-premium">
          <h2>Need Help Right Now?</h2>
          <p>Don't wait for the problem to get worse.</p>
          <a href="/contact/" class="cta-phone-premium">Call Now</a>
        </div>
      </div>
    </section>

    <!--Regional & Cost Info-- >
    <section class="section-premium" style="background: var(--color-gray-50);">
      <div class="container">
        <div class="content-wrapper">
          <h2>${content.sections.regionalVariation.title}</h2>
          <p>${parseMarkdown(content.sections.regionalVariation.content)}</p>
          
          <h2 style="margin-top: var(--space-12);">${content.sections.costFactors.title}</h2>
          <p>${parseMarkdown(content.sections.costFactors.content)}</p>
        </div>
      </div>
    </section>

    <!--Prevention Tips-- >
    <section class="section-premium">
      <div class="container">
        <div class="content-wrapper">
          <h2>${content.sections.preventionTips.title}</h2>
          <p>${content.sections.preventionTips.intro}</p>
          <div class="tips-grid">
            ${content.sections.preventionTips.tips.map(tip => `
              <div class="tip-card">
                <strong>Tip</strong>
                <p>${tip}</p>
              </div>
            `).join('')}
          </div>
          <p style="margin-top: var(--space-8); text-align: center;">${content.sections.preventionTips.outro}</p>
        </div>
      </div>
    </section>

    <!--Cities Grid-- >
    <section class="section-premium" id="cities" style="background: var(--color-gray-100);">
      <div class="container">
        <div class="section-header">
          <h2>Find ${service.name} in Your City</h2>
          <div class="section-divider"></div>
          <p>Select your city to connect with local professionals:</p>
        </div>
        <div class="cities-grid">
          ${cities.map(c => `<a href="/services/${service.slug}/${c.slug}/" class="city-link">${c.name}, ${c.state}</a>`).join('\n')}
        </div>
      </div>
    </section>

    <!--FAQs -->
    <section class="section-premium faq-section">
      <div class="container">
        <div class="faq-modern">
          <h2>Frequently Asked Questions</h2>
          <div class="faq-items">
            ${content.faqs.map(faq => `
              <div class="faq-item">
                <button class="faq-question" aria-expanded="false">
                  <span class="question-text">${faq.question}</span>
                  <span class="faq-icon">+</span>
                </button>
                <div class="faq-answer">
                  <p>${faq.answer}</p>
                </div>
              </div>
            `).join('')}
          </div>
        </div>
      </div>
    </section>

    <!--Disclaimer -->
      <section class="content-section">
        <div class="container">
          <div class="content-wrapper">
            ${getDisclaimer()}
          </div>
        </div>
      </section>
  </main >

      ${ getFooter(services, cities) }
</body >
</html > `;

    const outputPath = path.join(OUTPUT_DIR, 'services', service.slug, 'index.html');
    ensureDir(path.dirname(outputPath));
    fs.writeFileSync(outputPath, html);
  });

  console.log(`✓ Generated ${ services.length } PREMIUM service hub pages`);
}

module.exports = { generatePremiumServicePages };
