/**
 * HTML SITEMAP PAGE GENERATOR
 * ============================
 * Generates a user-friendly HTML sitemap page
 */

const fs = require('fs');
const path = require('path');

const services = require('../data/services.json');
const cities = require('../data/cities.json');
const blogArticles = require('../data/blog-articles.json');
const { getHeader, getNavigation, getFooter } = require('./seo-utils');

function generateHTMLSitemap(OUTPUT_DIR) {
  console.log('Generating HTML sitemap page...');

  const html = `${getHeader()}
  <title>Sitemap - All Pages | Rapid Service Connect</title>
  <meta name="description" content="Complete sitemap of Rapid Service Connect - Browse all services, cities, blog articles, and pages.">
  <link rel="canonical" href="http://rapidserviceconnect.com/sitemap/">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
  <link rel="stylesheet" href="/css/mega-menu.css">
</head>
<body>
  ${getNavigation(services, cities)}
  
  <main id="main-content">
    <div class="page-header">
      <div class="container">
        <h1>Complete Sitemap</h1>
        <p>Browse all pages on Rapid Service Connect</p>
      </div>
    </div>
    
    <section class="section">
      <div class="container">
        <div class="sitemap-grid">
          
          <!-- Services Section -->
          <div class="sitemap-section">
            <h2>🔧 Services (${services.length})</h2>
            <ul class="sitemap-links">
              ${services.map(service => `<li><a href="/${service.slug}/">${service.name}</a></li>`).join('\n              ')}
            </ul>
          </div>

          <!-- Cities Section -->
          <div class="sitemap-section">
            <h2>🏙️ Cities (${cities.length})</h2>
            <ul class="sitemap-links">
              ${cities.map(city => `<li><a href="/${city.slug}/">${city.name}, ${city.state}</a></li>`).join('\n              ')}
            </ul>
          </div>

          <!-- Blog Section -->
          <div class="sitemap-section">
            <h2>📝 Blog Articles (${blogArticles.length})</h2>
            <ul class="sitemap-links">
              ${blogArticles.map(article => `<li><a href="/blog/${article.slug}/">${article.title}</a></li>`).join('\n              ')}
            </ul>
          </div>

          <!-- Core Pages Section -->
          <div class="sitemap-section">
            <h2>📄 Core Pages</h2>
            <ul class="sitemap-links">
              <li><a href="/">Homepage</a></li>
              <li><a href="/about/">About Us</a></li>
              <li><a href="/how-it-works/">How It Works</a></li>
              <li><a href="/contact/">Contact</a></li>
              <li><a href="/blog/">Blog Index</a></li>
            </ul>
          </div>

          <!-- Legal Pages Section -->
          <div class="sitemap-section">
            <h2>⚖️ Legal Pages</h2>
            <ul class="sitemap-links">
              <li><a href="/privacy-policy/">Privacy Policy</a></li>
              <li><a href="/terms/">Terms of Use</a></li>
              <li><a href="/disclaimer/">Disclaimer</a></li>
            </ul>
          </div>

          <!-- XML Sitemaps Section -->
          <div class="sitemap-section">
            <h2>🗺️ XML Sitemaps</h2>
            <ul class="sitemap-links">
              <li><a href="/sitemap.xml">Main Sitemap Index</a></li>
              <li><a href="/sitemap-services.xml">Services Sitemap</a></li>
              <li><a href="/sitemap-cities.xml">Cities Sitemap</a></li>
              <li><a href="/sitemap-blog.xml">Blog Sitemap</a></li>
              <li><a href="/sitemap-all.xml">All Pages Sitemap</a></li>
            </ul>
          </div>

        </div>
      </div>
    </section>
  </main>
  
  ${getFooter(services, cities)}
</body>
</html>`;

  const outputPath = path.join(OUTPUT_DIR, 'sitemap', 'index.html');
  fs.mkdirSync(path.dirname(outputPath), { recursive: true });
  fs.writeFileSync(outputPath, html);

  console.log('✓ Generated HTML sitemap page at /sitemap/');
}

module.exports = { generateHTMLSitemap };
