// Page generation utility functions
const fs = require('fs');
const path = require('path');

function getHeader() {
  return `<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width, initial-scale=1.0">`;
}

function getNavigation(services, cities) {
  return `<header class="site-header">
    <div class="container header-container">
      <a href="/" class="site-logo">
        <span class="logo-text">Rapid Service Connect</span>
      </a>
      
      <button class="mobile-menu-toggle" aria-label="Toggle menu" aria-expanded="false">
        ☰
      </button>
      
      <nav class="main-nav">
        <ul class="nav-menu">
          <li class="nav-item">
            <a href="/" class="nav-link">Home</a>
          </li>
          <li class="nav-item">
            <a href="/how-it-works/" class="nav-link">How It Works</a>
          </li>
          <li class="nav-item has-dropdown">
            <a href="#" class="nav-link dropdown-toggle">Services</a>
            <ul class="dropdown-menu">
              ${services.map(s => `<li><a href="/services/${s.slug}/" class="nav-link">${s.name}</a></li>`).join('\n              ')}
            </ul>
          </li>
          <li class="nav-item has-dropdown">
            <a href="#" class="nav-link dropdown-toggle">Cities</a>
            <ul class="dropdown-menu">
              ${cities.map(c => `<li><a href="/cities/${c.slug}/" class="nav-link">${c.name}, ${c.state}</a></li>`).join('\n              ')}
            </ul>
          </li>
          <li class="nav-item">
            <a href="/blog/" class="nav-link">Blog</a>
          </li>
          <li class="nav-item">
            <a href="/contact/" class="nav-link">Contact</a>
          </li>
        </ul>
      </nav>
    </div>
  </header>`;
}

function getFooter(services, cities) {
  return `<footer class="site-footer">
    <div class="container">
      <div class="footer-content">
        <div class="footer-column">
          <h4>Company</h4>
          <ul class="footer-links">
            <li><a href="/about/">About Us</a></li>
            <li><a href="/how-it-works/">How It Works</a></li>
            <li><a href="/contact/">Contact</a></li>
          </ul>
        </div>
        
        <div class="footer-column">
          <h4>Services</h4>
          <ul class="footer-links">
            ${services.map(s => `<li><a href="/services/${s.slug}/">${s.name}</a></li>`).join('\n            ')}
          </ul>
        </div>
        
        <div class="footer-column">
          <h4>Legal</h4>
          <ul class="footer-links">
            <li><a href="/disclaimer/">Disclaimer</a></li>
            <li><a href="/privacy-policy/">Privacy Policy</a></li>
            <li><a href="/terms/">Terms of Use</a></li>
          </ul>
        </div>
        
        <div class="footer-column">
          <h4>Cities</h4>
          <ul class="footer-links">
            ${cities.slice(0, 5).map(c => `<li><a href="/services/plumbing/${c.slug}/">${c.name}, ${c.state}</a></li>`).join('\n            ')}
          </ul>
        </div>
      </div>
      
      <div class="footer-bottom">
        <p>&copy; ${new Date().getFullYear()} Rapid Service Connect. All rights reserved.</p>
      </div>
    </div>
  </footer>
  
  <script src="/js/main.js"></script>
</body>
</html>`;
}

function getDisclaimer() {
  return `<div class="disclaimer">
    <h4>Disclaimer:</h4>
    <p>
      Rapid Service Connect is an independent referral service that connects consumers with third-party service providers. 
      We do not directly provide plumbing, HVAC, electrical, towing, or restoration services. All service providers are 
      independent professionals. Availability and response times may vary by location. Calls may be recorded for quality 
      and compliance purposes.
    </p>
  </div>`;
}

module.exports = {
  generateHomepage,
  generateServiceHubPages,
  generateCityMoneyPages,
  generateCityHubPages,
  generateBlogPages,
  generateCorePages
};

// This file is too long - splitting into separate generator functions
// Homepage generator - imported at top of generate-pages.js
function generateHomepage(services, cities, OUTPUT_DIR, ensureDir) {
  const html = `${getHeader()}
  <title>24/7 Emergency Services – Instant Local Help | Rapid Service Connect</title>
  <meta name="description" content="Connect with 24/7 emergency services instantly. Verified plumbers, electricians, AC repair, water damage restoration, and towing in major US cities.">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
</head>
<body>
  ${getNavigation(services, cities)}
  
  <section class="hero">
    <div class="container hero-content">
      <h1>24/7 Emergency Services Near You</h1>
      <p>Instantly connect with verified professionals for plumbing, electrical, HVAC, water damage, and towing emergencies.</p>
      <div class="hero-buttons">
        <a href="#service-selector" class="btn btn-primary btn-large">Get Help Now</a>
        <a href="/how-it-works/" class="btn btn-secondary btn-large">How It Works</a>
      </div>
    </div>
  </section>
  
  <section class="section" id="service-selector">
    <div class="container">
      <div class="service-selector">
        <h2>Choose Your Service and City</h2>
        <p class="text-center mb-6">Select your emergency service and location to get immediate help</p>
        
        <form id="service-selector-form" class="selector-form">
          <div class="form-group">
            <label for="service-select">Select Service</label>
            <select id="service-select" class="form-select" required>
              <option value="">-- Choose a Service --</option>
              ${services.map(s => `<option value="${s.slug}">${s.name}</option>`).join('\n              ')}
            </select>
          </div>
          
          <div class="form-group">
            <label for="city-select">Select City</label>
            <select id="city-select" class="form-select" required>
              <option value="">-- Choose a City --</option>
              ${cities.map(c => `<option value="${c.slug}">${c.name}, ${c.state}</option>`).join('\n              ')}
            </select>
          </div>
          
          <button type="submit" class="btn btn-primary btn-large" style="width: 100%;">Connect Now</button>
        </form>
      </div>
    </div>
  </section>
  
  <section class="section">
    <div class="container">
      <h2 class="text-center mb-8">Emergency Services We Can Help You With</h2>
      <div class="services-grid">
        ${services.map(s => `<a href="/services/${s.slug}/" class="service-card">
          <span class="service-icon">${s.icon}</span>
          <h3>${s.name}</h3>
          <p>${s.description}</p>
        </a>`).join('\n        ')}
      </div>
    </div>
  </section>
  
  <section class="section" style="background: var(--color-gray-100);">
    <div class="container">
      <h2 class="text-center mb-8">Cities Currently Covered</h2>
      <div class="cities-grid">
        ${cities.map(c => `<a href="/cities/${c.slug}/" class="city-link">${c.name}, ${c.state}</a>`).join('\n        ')}
      </div>
    </div>
  </section>
  
  <section class="section">
    <div class="container">
      <h2 class="text-center mb-8">How It Works</h2>
      <div class="steps-grid">
        <div class="step-card">
          <div class="step-number">1</div>
          <h3>Choose Service & City</h3>
          <p>Select the emergency service you need and your location.</p>
        </div>
        <div class="step-card">
          <div class="step-number">2</div>
          <h3>Call the Number</h3>
          <p>Get connected instantly with pre-vetted local professionals.</p>
        </div>
        <div class="step-card">
          <div class="step-number">3</div>
          <h3>Get Help Fast</h3>
          <p>Speak directly with a qualified provider who can help immediately.</p>
        </div>
      </div>
    </div>
  </section>
  
  <section class="section" style="background: var(--color-gray-100);">
    <div class="container">
      <div class="content-wrapper">
        ${getDisclaimer()}
      </div>
    </div>
  </section>
  
  ${getFooter(services, cities)}`;

  ensureDir(OUTPUT_DIR);
  fs.writeFileSync(path.join(OUTPUT_DIR, 'index.html'), html);
  console.log('✓ Generated homepage');
}

function generateServiceHubPages(services, cities, blogArticles, OUTPUT_DIR, ensureDir) {
  services.forEach(service => {
    const relatedArticles = blogArticles.filter(a => a.service === service.id);
    const html = `${getHeader()}
  <title>${service.name} Near You – 24/7 Emergency Help | Rapid Service Connect</title>
  <meta name="description" content="Learn how Rapid Service Connect instantly links you with trusted 24/7 ${service.name.toLowerCase()} providers in major US cities.">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
</head>
<body>
  ${getNavigation(services, cities)}
  
  <div class="page-header">
    <div class="container">
      <h1>${service.name} – 24/7 Help Near You</h1>
      <p>Connect instantly with verified ${service.name.toLowerCase()} professionals in your city.</p>
    </div>
  </div>
  
  <section class="content-section">
    <div class="container">
      <div class="content-wrapper">
        <h2>Common ${service.name} Emergencies We Handle</h2>
        <ul>
          ${service.emergencies.map(e => `<li><strong>${e}</strong></li>`).join('\n          ')}
        </ul>
      </div>
    </div>
  </section>
  
  <section class="content-section" style="background: var(--color-gray-100);">
    <div class="container">
      <div class="content-wrapper">
        <h2>Cities We Currently Serve</h2>
        <div class="cities-grid">
          ${cities.map(c => `<a href="/services/${service.slug}/${c.slug}/" class="city-link">${c.name}, ${c.state}</a>`).join('\n          ')}
        </div>
      </div>
    </div>
  </section>
  
  ${getFooter(services, cities)}`;

    const outputPath = path.join(OUTPUT_DIR, 'services', service.slug, 'index.html');
    ensureDir(path.dirname(outputPath));
    fs.writeFileSync(outputPath, html);
  });
  console.log(`✓ Generated ${services.length} service hub pages`);
}

// Due to file length limits, splitting city money page generator
function generateCityMoneyPages(services, cities, blogArticles, OUTPUT_DIR, ensureDir) {
  let count = 0;
  services.forEach(service => {
    cities.forEach(city => {
      const trackingNumber = `[[TRACKING_NUMBER_${service.id.toUpperCase()}_${city.id.toUpperCase()}]]`;
      const html = `${getHeader()}
  <title>Emergency ${service.name} in ${city.name} – 24/7 Fast Help | Rapid Service Connect</title>
  <meta name="description" content="Need ${service.name.toLowerCase()} in ${city.name}? Call now to instantly connect with verified 24/7 professionals near you.">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
</head>
<body>
  ${getNavigation(services, cities)}
  
  <div class="page-header">
    <div class="container">
      <h1>Emergency ${service.name} in ${city.name} – 24/7 Fast Help</h1>
      <p>Get connected with qualified ${service.name.toLowerCase()} professionals in ${city.name}, ${city.state} immediately.</p>
    </div>
  </div>
  
  <section class="section">
    <div class="container">
      <div class="cta-block">
        <h2>Call Now for Immediate Help in ${city.name}</h2>
        <a href="tel:${trackingNumber}" class="cta-phone">${trackingNumber}</a>
        <p>Available 24/7 • Local ${city.name} Professionals • Fast Response</p>
      </div>
    </div>
  </section>
  
  <section class="content-section">
    <div class="container">
      <div class="content-wrapper">
        <h2>Types of ${service.name} Emergencies in ${city.name}</h2>
        <ul>
          ${service.emergencies.map(e => `<li><strong>${e}</strong> – Immediate response available</li>`).join('\n          ')}
        </ul>
      </div>
    </div>
  </section>
  
  <section class="content-section" style="background: var(--color-gray-100);">
    <div class="container">
      <div class="content-wrapper">
        ${getDisclaimer()}
      </div>
    </div>
  </section>
  
  ${getFooter(services, cities)}`;

      const outputPath = path.join(OUTPUT_DIR, 'services', service.slug, city.slug, 'index.html');
      ensureDir(path.dirname(outputPath));
      fs.writeFileSync(outputPath, html);
      count++;
    });
  });
  console.log(`✓ Generated ${count} city money pages`);
}

function generateCityHubPages(services, cities, OUTPUT_DIR, ensureDir) {
  cities.forEach(city => {
    const html = `${getHeader()}
  <title>Emergency Services in ${city.name} – 24/7 Local Help | Rapid Service Connect</title>
  <meta name="description" content="Connect with 24/7 emergency services in ${city.name}, ${city.state}. Instant access to plumbers, electricians, AC repair, water damage restoration, and towing.">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
</head>
<body>
  ${getNavigation(services, cities)}
  
  <div class="page-header">
    <div class="container">
      <h1>Emergency Services in ${city.name}</h1>
      <p>24/7 local help for residents of ${city.name}, ${city.stateFullName}</p>
    </div>
  </div>
  
  <section class="section">
    <div class="container">
      <h2 class="text-center mb-8">Available Services in ${city.name}</h2>
      <div class="services-grid">
        ${services.map(s => `<a href="/services/${s.slug}/${city.slug}/" class="service-card">
          <span class="service-icon">${s.icon}</span>
          <h3>${s.name}</h3>
          <p>Click for immediate ${s.name.toLowerCase()} help in ${city.name}</p>
        </a>`).join('\n        ')}
      </div>
    </div>
  </section>
  
  ${getFooter(services, cities)}`;

    const outputPath = path.join(OUTPUT_DIR, 'cities', city.slug, 'index.html');
    ensureDir(path.dirname(outputPath));
    fs.writeFileSync(outputPath, html);
  });
  console.log(`✓ Generated ${cities.length} city hub pages`);
}

function generateBlogPages(services, cities, blogArticles, OUTPUT_DIR, ensureDir) {
  // Blog index
  const blogIndexHtml = `${getHeader()}
  <title>Emergency Services Blog – Tips & Guides | Rapid Service Connect</title>
  <meta name="description" content="Expert advice for handling home emergencies. Plumbing, electrical, HVAC, water damage, and towing tips.">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
</head>
<body>
  ${getNavigation(services, cities)}
  
  <div class="page-header">
    <div class="container">
      <h1>Emergency Services Blog</h1>
      <p>Expert tips and guides for handling emergencies</p>
    </div>
  </div>
  
  <section class="section">
    <div class="container">
      <div class="content-wrapper">
        ${services.map(service => {
    const serviceArticles = blogArticles.filter(a => a.service === service.id);
    return `<h3>${service.name} Articles</h3>
        <ul class="mb-8">
          ${serviceArticles.map(a => `<li><a href="/blog/${a.slug}/">${a.title}</a></li>`).join('\n          ')}
        </ul>`;
  }).join('\n')}
      </div>
    </div>
  </section>
  
  ${getFooter(services, cities)}`;

  ensureDir(path.join(OUTPUT_DIR, 'blog'));
  fs.writeFileSync(path.join(OUTPUT_DIR, 'blog', 'index.html'), blogIndexHtml);

  // Individual articles
  blogArticles.forEach(article => {
    const service = services.find(s => s.id === article.service);
    const html = `${getHeader()}
  <title>${article.title}</title>
  <meta name="description" content="${article.metaDescription}">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
</head>
<body>
  ${getNavigation(services, cities)}
  
  <article class="blog-article">
    <div class="container">
      <h1>${article.title}</h1>
      <div class="article-meta">Category: <a href="/services/${service.slug}/">${service.name}</a></div>
      
      <h2>Understanding the Situation</h2>
      <p>When facing ${service.name.toLowerCase()} emergencies, knowing what to do can make all the difference.</p>
      
      <div class="cta-block mt-12">
        <h2>Need Immediate ${service.name} Help?</h2>
        <p>Connect with a 24/7 provider in your city right now.</p>
        <a href="/services/${service.slug}/" class="btn btn-primary btn-large">Get Help Now</a>
      </div>
      
      <div class="mt-12">${getDisclaimer()}</div>
    </div>
  </article>
  
  ${getFooter(services, cities)}`;

    ensureDir(path.join(OUTPUT_DIR, 'blog', article.slug));
    fs.writeFileSync(path.join(OUTPUT_DIR, 'blog', article.slug, 'index.html'), html);
  });
  console.log(`✓ Generated ${blogArticles.length + 1} blog pages`);
}

function generateCorePages(services, cities, OUTPUT_DIR, ensureDir) {
  // About
  ensureDir(path.join(OUTPUT_DIR, 'about'));
  fs.writeFileSync(path.join(OUTPUT_DIR, 'about', 'index.html'), `${getHeader()}
  <title>About Rapid Service Connect</title>
  <meta name="description" content="Learn about Rapid Service Connect and how we connect you with trusted emergency service professionals.">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
</head>
<body>
  ${getNavigation(services, cities)}
  <div class="page-header"><div class="container"><h1>About Rapid Service Connect</h1></div></div>
  <section class="content-section"><div class="container"><div class="content-wrapper">
    <h2>Who We Are</h2>
    <p>Independent referral service connecting people with qualified emergency professionals 24/7.</p>
    ${getDisclaimer()}
  </div></div></section>
  ${getFooter(services, cities)}`);

  //How It Works - Comprehensive Version
  ensureDir(path.join(OUTPUT_DIR, 'how-it-works'));
  fs.writeFileSync(path.join(OUTPUT_DIR, 'how-it-works', 'index.html'), `${getHeader()}
  <title>How It Works – Get Emergency Help in 3 Simple Steps | Rapid Service Connect</title>
  <meta name="description" content="Learn how to get emergency service help fast. Choose your service, call the number, and connect with local professionals instantly.">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
  <link rel="stylesheet" href="/css/mega-menu.css">
  <link rel="stylesheet" href="/css/chatbot.css">
</head>
<body>
  ${getNavigation(services, cities)}
  
  <main id="main-content">
    <div class="page-header">
      <div class="container">
        <h1>How It Works</h1>
        <p>Getting emergency help is fast, simple, and straightforward</p>
      </div>
    </div>
    
    <section class="section">
      <div class="container">
        <div class="steps-grid">
          <div class="step-card">
            <div class="step-number">1</div>
            <h3>🔍 Choose Service & Location</h3>
            <p><strong>Find Your Service:</strong> Browse our ${services.length} emergency services including plumbing, electrical, HVAC, water damage, towing, and more.</p>
            <p><strong>Select Your City:</strong> We cover ${cities.length} major U.S. cities. Find your location from our comprehensive list.</p>
            <p><strong>Quick Navigation:</strong> Use our homepage service selector or browse by category for instant access.</p>
          </div>
          
          <div class="step-card">
            <div class="step-number">2</div>
            <h3>📞 Call the Number Provided</h3>
            <p><strong>Instant Connection:</strong> Each service page shows a dedicated phone number that connects you directly with local professionals.</p>
            <p><strong>24/7 Availability:</strong> Our network operates around the clock, 365 days a year for true emergency response.</p>
            <p><strong>No Forms or Waiting:</strong> Skip the paperwork – just call and speak with a real professional immediately.</p>
          </div>
          
          <div class="step-card">
            <div class="step-number">3</div>
            <h3>✅ Get Professional Help Fast</h3>
            <p><strong>Verified Professionals:</strong> Connect with licensed, insured service providers in your area.</p>
            <p><strong>Fast Response:</strong> Most professionals can respond within hours, with many offering same-day emergency service.</p>
            <p><strong>Local Experts:</strong> Work with professionals who know your area and understand local codes and requirements.</p>
          </div>
        </div>
      </div>
    </section>
    
    <section class="section" style="background: var(--color-gray-100);">
      <div class="container">
        <h2 class="text-center mb-8">Why Choose Rapid Service Connect?</h2>
        <div class="services-grid">
          <div class="service-card">
            <span class="service-icon">⚡</span>
            <h3>Instant Access</h3>
            <p>No waiting for callbacks or form submissions. Get connected to professionals in seconds, not hours.</p>
          </div>
          
          <div class="service-card">
            <span class="service-icon">🏆</span>
            <h3>Pre-Vetted Network</h3>
            <p>All professionals in our network are licensed, insured, and experienced in emergency services.</p>
          </div>
          
          <div class="service-card">
            <span class="service-icon">🌍</span>
            <h3>Wide Coverage</h3>
            <p>Serving ${cities.length} major cities across the United States with ${services.length} different service categories.</p>
          </div>
          
          <div class="service-card">
            <span class="service-icon">🆓</span>
            <h3>Free Referral Service</h3>
            <p>Our connection service is completely free. You only pay the professional directly for their services.</p>
          </div>
          
          <div class="service-card">
            <span class="service-icon">🕐</span>
            <h3>24/7 Availability</h3>
            <p>Emergencies don't wait for business hours. Neither do we. Get help any time, day or night.</p>
          </div>
          
          <div class="service-card">
            <span class="service-icon">🔒</span>
            <h3>Trusted & Transparent</h3>
            <p>Clear information, honest connections, and no hidden fees. What you see is what you get.</p>
          </div>
        </div>
      </div>
    </section>
    
    <section class="section">
      <div class="container">
        <div class="content-wrapper">
          <h2>What To Expect When You Call</h2>
          <p>When you call the number on any of our service pages, here's what happens:</p>
          
          <ol>
            <li><strong>Immediate Answer:</strong> A professional service provider or dispatch team answers your call promptly.</li>
            <li><strong>Service Verification:</strong> They'll confirm the type of emergency service you need and your location.</li>
            <li><strong>Professional Assessment:</strong> The provider will ask relevant questions to understand your situation.</li>
            <li><strong>Fast Scheduling:</strong> For emergencies, most providers can dispatch immediately or within hours.</li>
            <li><strong>Clear Pricing:</strong> You'llget upfront information about pricing and what to expect.</li>
            <li><strong>Expert Service:</strong> A qualified professional arrives to solve your problem quickly and correctly.</li>
          </ol>
          
          <h2 class="mt-12">Available Services</h2>
          <p>We connect you with professionals for all major emergency services:</p>
          
          <div class="cities-grid mt-6">
            ${services.map(s => `<a href="/services/${s.slug}/" class="city-link">${s.icon} ${s.name}</a>`).join('\n            ')}
          </div>
          
          <h2 class="mt-12">Cities We Serve</h2>
          <p>Our network covers ${cities.length} major metropolitan areas:</p>
          
          <div class="cities-grid mt-6">
            ${cities.slice(0, 20).map(c => `<a href="/cities/${c.slug}/" class="city-link">${c.name}, ${c.state}</a>`).join('\n            ')}
            <a href="/" class="city-link" style="background: var(--gradient-primary); color: white; font-weight: bold;">+ ${cities.length - 20} More Cities</a>
          </div>
        </div>
      </div>
    </section>
    
    <section class="section faq-section" style="background: var(--color-gray-100);">
      <div class="container">
        <div class="content-wrapper">
          <h2>Frequently Asked Questions</h2>
          
          <div class="faq-item">
            <button class="faq-question">How quickly can I get help?</button>
            <div class="faq-answer"><p>Most professionals can respond within hours. For true emergencies like burst pipes or electrical hazards, many providers offer same-day or even same-hour service depending on availability in your area.</p></div>
          </div>
          
          <div class="faq-item">
            <button class="faq-question">Do you provide the services directly?</button>
            <div class="faq-answer"><p>No, we are a referral service. We connect you with independent, licensed professionals who provide the actual services. This allows us to maintain a large network of specialists across many cities.</p></div>
          </div>
          
          <div class="faq-item">
            <button class="faq-question">Is there a fee to use Rapid Service Connect?</button>
            <div class="faq-answer"><p>Our referral service is completely free. You only pay the service professional directly for their work. Pricing is determined by the provider based on the scope of your specific job.</p></div>
          </div>
          
          <div class="faq-item">
            <button class="faq-question">Are the professionals licensed and insured?</button>
            <div class="faq-answer"><p>Yes, all professionals in our network are required to maintain proper licensing and insurance for their trade. However, we recommend confirming their credentials when they arrive for added peace of mind.</p></div>
          </div>
          
          <div class="faq-item">
            <button class="faq-question">What if I'm not in one of your listed cities?</button>
            <div class="faq-answer"><p>Our network primarily covers the ${cities.length} major metropolitan areas listed on our site. If you're in a nearby suburb or surrounding area, call anyway – many providers serve broader regions.</p></div>
          </div>
          
          <div class="faq-item">
            <button class="faq-question">Can I schedule non-emergency service?</button>
            <div class="faq-answer"><p>While we specialize in emergency connections, many professionals also handle scheduled maintenance and non-urgent work. Simply call and discuss your needs with the provider.</p></div>
          </div>
        </div>
      </div>
    </section>
    
    <section class="section">
      <div class="container">
        <div class="cta-block">
          <h2>Ready to Get Help?</h2>
          <p>Choose your service and location to connect with local professionals instantly</p>
          <a href="/" class="btn btn-primary btn-large">Find Your Service</a>
        </div>
        
        <div class="content-wrapper mt-12">
          ${getDisclaimer()}
        </div>
      </div>
    </section>
  </main>
  
  ${getFooter(services, cities)}`);

  // Contact
  ensureDir(path.join(OUTPUT_DIR, 'contact'));
  fs.writeFileSync(path.join(OUTPUT_DIR, 'contact', 'index.html'), `${getHeader()}
  <title>Contact Us – Rapid Service Connect</title>
  <meta name="description" content="Contact Rapid Service Connect for general inquiries or emergency service help.">
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
</head>
<body>
  ${getNavigation(services, cities)}
  <div class="page-header"><div class="container"><h1>Contact</h1></div></div>
  <section class="content-section"><div class="container"><div class="content-wrapper">
    <h2>Need Emergency Service?</h2>
    <p>Use our <a href="/">service selector</a> for immediate help.</p>
  </div></div></section>
  ${getFooter(services, cities)}`);

  // Legal pages
  ensureDir(path.join(OUTPUT_DIR, 'disclaimer'));
  ensureDir(path.join(OUTPUT_DIR, 'privacy-policy'));
  ensureDir(path.join(OUTPUT_DIR, 'terms'));

  fs.writeFileSync(path.join(OUTPUT_DIR, 'disclaimer', 'index.html'), `${getHeader()}
  <title>Disclaimer</title>
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
</head>
<body>
  ${getNavigation(services, cities)}
  <div class="page-header"><div class="container"><h1>Disclaimer</h1></div></div>
  <section class="content-section"><div class="container"><div class="content-wrapper">
    ${getDisclaimer()}
  </div></div></section>
  ${getFooter(services, cities)}`);

  fs.writeFileSync(path.join(OUTPUT_DIR, 'privacy-policy', 'index.html'), `${getHeader()}
  <title>Privacy Policy</title>
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
</head>
<body>
  ${getNavigation(services, cities)}
  <div class="page-header"><div class="container"><h1>Privacy Policy</h1></div></div>
  <section class="content-section"><div class="container"><div class="content-wrapper">
    <p>We collect information to connect you with service providers and improve our service.</p>
  </div></div></section>
  ${getFooter(services, cities)}`);

  fs.writeFileSync(path.join(OUTPUT_DIR, 'terms', 'index.html'), `${getHeader()}
  <title>Terms of Use</title>
  <link rel="stylesheet" href="/css/design-system.css">
  <link rel="stylesheet" href="/css/components.css">
  <link rel="stylesheet" href="/css/pages.css">
</head>
<body>
  ${getNavigation(services, cities)}
  <div class="page-header"><div class="container"><h1>Terms of Use</h1></div></div>
  <section class="content-section"><div class="container"><div class="content-wrapper">
    <p>Rapid Service Connect is a referral service. By using our service, you agree to these terms.</p>
  </div></div></section>
  ${getFooter(services, cities)}`);

  console.log('✓ Generated 6 core pages');
}
