/**
 * RAPID SERVICE CONNECT - MAIN JAVASCRIPT
 * ========================================
 * Handles mega menus, mobile navigation, and interactive elements
 */

// ===========================================
// MOBILE MENU TOGGLE
// ===========================================
const mobileMenuToggle = document.querySelector('.mobile-menu-toggle');
const mainNav = document.querySelector('.main-nav');

if (mobileMenuToggle && mainNav) {
  mobileMenuToggle.addEventListener('click', () => {
    const isExpanded = mobileMenuToggle.getAttribute('aria-expanded') === 'true';

    mobileMenuToggle.setAttribute('aria-expanded', !isExpanded);
    mainNav.classList.toggle('active');

    // Prevent body scroll when menu is open
    if (!isExpanded) {
      document.body.style.overflow = 'hidden';
    } else {
      document.body.style.overflow = '';
    }
  });
}

// ===========================================
// MOBILE ACCORDION FUNCTIONALITY
// ===========================================
const accordionHeaders = document.querySelectorAll('.mobile-accordion-header');

accordionHeaders.forEach(header => {
  header.addEventListener('click', () => {
    const content = header.nextElementSibling;
    const toggle = header.querySelector('.accordion-toggle');

    // Toggle active class
    content.classList.toggle('active');
    toggle.classList.toggle('active');
  });
});

// ===========================================
// SERVICE SELECTOR FORM (Homepage)
// ===========================================
const serviceSelectorForm = document.getElementById('service-selector-form');

if (serviceSelectorForm) {
  serviceSelectorForm.addEventListener('submit', (e) => {
    e.preventDefault();

    const serviceSelect = document.getElementById('service-select');
    const citySelect = document.getElementById('city-select');

    const service = serviceSelect.value;
    const city = citySelect.value;

    if (service && city) {
      // Redirect to city money page
      window.location.href = `/${service}-${city}/`;
    } else if (service) {
      // Redirect to service hub
      window.location.href = `/${service}/`;
    } else if (city) {
      // Redirect to city hub
      window.location.href = `/${city}/`;
    } else {
      alert('Please select a service and city');
    }
  });
}

// ===========================================
// FAQ ACCORDION (supports both standard and premium)
// ===========================================
const faqQuestions = document.querySelectorAll('.faq-question');

faqQuestions.forEach(question => {
  question.addEventListener('click', () => {
    const faqItem = question.closest('.faq-item');
    const answer = question.nextElementSibling;
    const isExpanded = faqItem.classList.contains('active');

    // Toggle current FAQ
    if (isExpanded) {
      faqItem.classList.remove('active');
      question.setAttribute('aria-expanded', 'false');
    } else {
      faqItem.classList.add('active');
      question.setAttribute('aria-expanded', 'true');
    }
  });
});

// ===========================================
// STICKY HEADER ON SCROLL
// ===========================================
let lastScroll = 0;
const siteHeader = document.querySelector('.site-header');

if (siteHeader) {
  window.addEventListener('scroll', () => {
    const currentScroll = window.pageYOffset;

    if (currentScroll > 100) {
      siteHeader.classList.add('scrolled');
    } else {
      siteHeader.classList.remove('scrolled');
    }

    lastScroll = currentScroll;
  });
}

// ===========================================
// CLOSE MOBILE MENU ON LINK CLICK
// ===========================================
const navLinks = document.querySelectorAll('.nav-link, .mobile-accordion-content a');

navLinks.forEach(link => {
  link.addEventListener('click', () => {
    if (window.innerWidth <= 768 && mainNav && mobileMenuToggle) {
      mainNav.classList.remove('active');
      mobileMenuToggle.setAttribute('aria-expanded', 'false');
      document.body.style.overflow = '';
    }
  });
});

// ===========================================
// MEGA MENU ACCESSIBILITY (Keyboard Navigation)
// ===========================================
const megaMenuItems = document.querySelectorAll('.nav-item.has-mega-menu');

megaMenuItems.forEach(item => {
  const link = item.querySelector('.nav-link');
  const megaMenu = item.querySelector('.mega-menu');

  if (link && megaMenu) {
    // Prevent navigation on click for desktop (mega menu should show on hover)
    link.addEventListener('click', (e) => {
      if (window.innerWidth > 768) {
        e.preventDefault();
        // Toggle mega menu visibility on click for better mobile/touch support
        const isVisible = megaMenu.style.visibility === 'visible';
        if (isVisible) {
          megaMenu.style.opacity = '';
          megaMenu.style.visibility = '';
          megaMenu.style.transform = '';
        } else {
          megaMenu.style.opacity = '1';
          megaMenu.style.visibility = 'visible';
          megaMenu.style.transform = 'translateY(0)';
        }
      }
    });

    // Open on focus for keyboard users
    link.addEventListener('focus', () => {
      if (window.innerWidth > 768) {
        megaMenu.style.opacity = '1';
        megaMenu.style.visibility = 'visible';
        megaMenu.style.transform = 'translateY(0)';
      }
    });

    // Close when focus leaves the mega menu
    item.addEventListener('focusout', (e) => {
      if (window.innerWidth > 768 && !item.contains(e.relatedTarget)) {
        megaMenu.style.opacity = '';
        megaMenu.style.visibility = '';
        megaMenu.style.transform = '';
      }
    });
  }
});

// ===========================================
// CLOSE MEGA MENU ON ESC KEY
// ===========================================
document.addEventListener('keydown', (e) => {
  if (e.key === 'Escape') {
    const megaMenus = document.querySelectorAll('.mega-menu');
    megaMenus.forEach(menu => {
      menu.style.opacity = '';
      menu.style.visibility = '';
      menu.style.transform = '';
    });
  }
});

// ===========================================
// SMOOTH SCROLL FOR ANCHOR LINKS
// ===========================================
const anchorLinks = document.querySelectorAll('a[href^="#"]');

anchorLinks.forEach(link => {
  link.addEventListener('click', (e) => {
    const href = link.getAttribute('href');

    if (href !== '#') {
      e.preventDefault();
      const target = document.querySelector(href);

      if (target) {
        target.scrollIntoView({
          behavior: 'smooth',
          block: 'start'
        });
      }
    }
  });
});

// ===========================================
// PERFORMANCE: Lazy Load Images (if any)
// ===========================================
if ('IntersectionObserver' in window) {
  const lazyImages = document.querySelectorAll('img[data-src]');

  const imageObserver = new IntersectionObserver((entries, observer) => {
    entries.forEach(entry => {
      if (entry.isIntersecting) {
        const img = entry.target;
        img.src = img.dataset.src;
        img.removeAttribute('data-src');
        imageObserver.unobserve(img);
      }
    });
  });

  lazyImages.forEach(img => imageObserver.observe(img));
}

// ===========================================
// DEBUG: Service Card Click Tracking
// ===========================================
const serviceCards = document.querySelectorAll('.service-card');
serviceCards.forEach(card => {
  card.addEventListener('click', (e) => {
    const href = card.getAttribute('href');
    console.log('✅ Service card clicked! Navigating to:', href);
    // Let the default behavior happen
  });
});

console.log('✅ Rapid Service Connect - Navigation System Loaded');
console.log(`📊 Found ${serviceCards.length} service cards on this page`);
